/*****************************************************************************/
/**
@file         FS300_Basic_Measurement.cpp
@copyright    (c) Rohde & Schwarz GmbH & Co. KG, Munich
@version	   Revision: 1.2, 08/2004                   

@author       W. Blanz
@responsible  W. Blanz
*
@language     C++ (maybe C also)
*
@description	Sample to set the FS300 Spectrum Analyser
				perform a measurement:
				Start Frequency :  10 MHz
				Stop  Frequency :  30 MHz				
				RBW				:	Auto
				VBW				:	Auto
				Reference Level	:	-10 dBm
				Sweep Points:	:	100
				

see          
*
history

	05 Dez 2003:	1.0 Initial version
	12 May 2004		1.1 Changed parameters
*
info
	The example software was tested with the revision 1.8 of the
	rssifs VXI PnP driver and FS300 Release 2.5

*
*  
*
*******************************************************************************
******************************************************************************/

#include "stdafx.h"
#include <stdio.h>
#include <rssifs.h>
#include <math.h>


ViRsrc resourceName				= "USB::0xAAD::0x6::100196";	//"Analyser1"; /* !! change to your FS300 serial number or logical device name !! */

ViSession instrumentHandle;			/* VISA instrument handle */
ViStatus Status=0;					/* VISA status */
ViChar message[256];				/* VISA message buffer */

ViReal64 referenceLevel		=-10;	/* Reference Level		:	-27 dBm */
ViInt32 frequencyParameter	=0;		/* Frequency Parameter	:	0 - Start And Stop */
ViReal64 startFrequency		=10e6;	/* Start Frequency		:	10 MHz */
ViReal64 stopFrequency		=30e6;	/* Stop Frequency		:	30 MHz */
ViInt32 resolutionBandwidth	=0;		/* Resolution Bandwidth	:	0 - Auto */
ViInt32 videoBandwidth		=0;		/* Video Bandwidth		:	0 - Auto  */
ViReal64 bandWidth			=0;		/* Bandwidth reading value */
ViInt32 sweepPoints			=100;	/* Number of Sweep Points per Trace */
ViReal64 _VI_FAR traceData[2048];	/* Trace Data Buffer */
ViInt32 samplesReturned	=0;		/* Samples returned form the FS300 */
  
/*======================================================================================================*/
/* Function:	void Error_Message(ViStatus Status)														*/
/* Purpose:		prints visa error and device error to screen											*/
/* Parameter:   ViStatus Status	- Status of last function call											*/
/* Return:		none																					*/
/*======================================================================================================*/
void Error_Message(ViStatus Status)
{
if(Status){ /* display device/driver errors */		
	printf("(Error Code: 0x%x)",Status);	
	Status= rssifs_error_message (instrumentHandle, Status, &message[0]);
	Status= rssifs_error_query (instrumentHandle, &Status, &message[0]);
	printf(&message[0]);		
	printf("\n");
	}	

}


/*======================================================================================================*/
/* Function:	void Volt2dBm(ViReal64 Volt)															*/
/* Purpose:		Calculate Volt to dBm																	*/
/* Parameter:   ViReal64 Volt	- Voltage Level															*/
/* Return:		ViReal64		- dBm Level																					*/
/*======================================================================================================*/
ViReal64 Volt2dBm(ViReal64 Volt)
{
	return(10*log10((pow(Volt,2)/(50*0.001))));
}

/*======================================================================================================*/
/* Function:	int main(void)																			*/
/* Purpose:		initialisation and main device settings													*/
/* Parameter:   																						*/
/* Return:		 0 if no error																			*/
/*				-1 if device open error																	*/
/*======================================================================================================*/

int main(void)
{
	/* Initialise Device */
	printf("*******************************************************\n");
	printf("* Open FS300 Resource String: %s *\n",resourceName);
	printf("*******************************************************\n");
	

	/* initialise device with reset */
	Status=rssifs_init (resourceName, VI_TRUE, VI_TRUE,&instrumentHandle);
	Error_Message(Status);
	if (Status)
		return(-1);
	else if(!Status)
		printf("\nDevice opened\n");

	/* Set Reference Level */
	rssifs_confRefLevel ( instrumentHandle,referenceLevel);
	Error_Message(Status);	
	if(!Status)
	{		
		Status=rssifs_getReferenceLevel (instrumentHandle,&referenceLevel);
		printf("Reference Level:	%f dBm\n",referenceLevel);		
	}


	/* Set Start and Stop Frequency */
	Status=rssifs_confStartStopFrq (instrumentHandle, frequencyParameter, startFrequency, stopFrequency);
	Error_Message(Status);	
	if(!Status)
	{		
		Status=rssifs_getStartFrequency (instrumentHandle,&startFrequency);
		printf("Start Frequency:	%f Hz\n",startFrequency);
		Status=rssifs_getStopFrequency (instrumentHandle,&stopFrequency);
		printf("Stop  Frequency:	%f Hz\n",stopFrequency);
	}
 
	/* Set RBW and VBW to AUTO Mode */
	Status=rssifs_configureBandwidth (instrumentHandle, resolutionBandwidth, videoBandwidth);
	Error_Message(Status);
	if(!Status)
	{		
		Status= rssifs_getResolutionBandwidth (instrumentHandle, &bandWidth);
		printf("Resolution Bandwidth:	%f Hz\n",bandWidth);
		Status= rssifs_getVideoBandwidth (instrumentHandle, &bandWidth);
		printf("Video Bandwidth     :	%f Hz\n",bandWidth);
	}

	/* Stop actual Sweep */
	Status=rssifs_actAbort (instrumentHandle);
	Error_Message(Status);

	/* Set Sweep Point */
	Status=rssifs_confSweepPoints (instrumentHandle, sweepPoints);
	Error_Message(Status);
	if(!Status)
	{		
		Status= rssifs_getSweepPoints (instrumentHandle,&sweepPoints);
		printf("Sweep Points:		%d \n",sweepPoints);		
	}

	/* Start Actual Sweep */
	Status=rssifs_actSendTrg (instrumentHandle);
	Error_Message(Status);

	/* Read One Complete Trace */
	Status=rssifs_readCompleteSweepData (instrumentHandle, traceData, &samplesReturned);
	Error_Message(Status);
	if(!Status)
	{			
		printf("Samples returned:	%d \n",samplesReturned);		
	}

#if 0	/* set to 1 to enable trace data output */
	if(!Status)
	{		
		printf("Samples : %d \n",samplesReturned);
		for (int loop=0;loop<samplesReturned;loop++)
		{
			printf("Sample[%d]:			%f (%f dBm)\n",loop,traceData[loop],Volt2dBm(traceData[loop]));
		}
		printf("\n\n");
	}
#endif

	/* Device Configured */
	printf("\nDevice Configured!\n");
	
	/* Close Device */
	Status=rssifs_close (instrumentHandle);
	return 0;
}


